/*:
 * @target MZ
 * @plugindesc v1.0 タイトル回想（コモンイベント再生）/ 解放管理 / 多言語連動（HS_LangSimple互換）
 * @author You
 *
 * @help
 * ■ 使い方（最短）
 * 1) プラグインを導入してパラメータの「Scenes」に回想したいコモンイベントを登録。
 * 2) 各シーンの本編イベント末尾にプラグインコマンド「シーン解放」を1行置く。
 *    → 一度見たら回想メニューで選べるようになります（グローバル保存）。
 * 3) 回想再生後は、プラグインコマンド「回想からタイトルへ戻る」で帰還。
 *
 * ■ 回想フラグスイッチ
 * 回想再生中は指定スイッチ（RecallFlagSwitchId）が自動ONになります。
 * 回想用の簡略演出や報酬スキップ等の分岐に使ってください。
 *
 * ■ 言語連動
 * 「LanguageVariableId」にHS_LangSimpleで使っている言語変数IDを入れてください。
 * 0:日本語 / 1:English / 2:简体中文 / 3:한국어
 * タイトルの「回想」表示やシーン名、ロック時の文言が言語ごとに切り替わります。
 *
 * ■ 永続保存
 * 解放済みシーンはConfig（config.rpgsave）に保存され、セーブデータを跨いで保持されます。
 *
 * ------------------------------------------------------------
 * @param LanguageVariableId
 * @text 言語設定の変数ID
 * @type variable
 * @desc 0:JP / 1:EN / 2:ZH / 3:KO（HS_LangSimpleで使用中の言語変数ID）
 * @default 0
 *
 * @param PutInTitle
 * @text タイトルに回想を追加
 * @type boolean
 * @default true
 *
 * @param RecallCommandJP
 * @text 回想コマンド名(JP)
 * @type string
 * @default 回想
 *
 * @param RecallCommandEN
 * @text 回想コマンド名(EN)
 * @type string
 * @default Memories
 *
 * @param RecallCommandZH
 * @text 回想コマンド名(ZH)
 * @type string
 * @default 回想
 *
 * @param RecallCommandKO
 * @text 回想コマンド名(KO)
 * @type string
 * @default 회상
 *
 * @param LockedTextJP
 * @text 未解放表示(JP)
 * @type string
 * @default ？？？
 *
 * @param LockedTextEN
 * @text 未解放表示(EN)
 * @type string
 * @default ????
 *
 * @param LockedTextZH
 * @text 未解放表示(ZH)
 * @type string
 * @default ？？？
 *
 * @param LockedTextKO
 * @text 未解放表示(KO)
 * @type string
 * @default ？？？
 *
 * @param RecallFlagSwitchId
 * @text 回想フラグ用スイッチ
 * @type switch
 * @default 0
 * @desc 回想再生中は自動ON。イベント内で分岐に使えます（再生終了時は手動で戻ります）。
 *
 * @param RecallMapId
 * @text 回想再生用マップID
 * @type number
 * @min 1
 * @default 1
 * @desc 回想再生を開始するマップ。黒背景の専用マップを1枚用意するのがおすすめ。
 *
 * @param RecallStartX
 * @text 回想開始X座標
 * @type number
 * @min 0
 * @default 0
 *
 * @param RecallStartY
 * @text 回想開始Y座標
 * @type number
 * @min 0
 * @default 0
 *
 * @param Scenes
 * @text シーン一覧
 * @type struct<HSRecallScene>[]
 * @default []
 *
 * @command UnlockScene
 * @text シーン解放
 * @desc 指定IDのシーンを解放状態にします（Configに保存）
 * @arg id
 * @text シーンID
 * @type string
 *
 * @command UnlockByCommonEvent
 * @text コモンイベントIDで解放
 * @desc 指定のコモンイベントを持つシーンを解放
 * @arg commonEventId
 * @text コモンイベントID
 * @type common_event
 *
 * @command OpenRecall
 * @text 回想メニューを開く
 * @desc タイトル以外からも呼べます（開発/デバッグ用）
 *
 * @command ExitToTitleFromRecall
 * @text 回想からタイトルへ戻る
 * @desc 回想イベントの最後で実行してください（終了後はタイトルへ）
 */
/*~struct~HSRecallScene:
 * @param id
 * @text シーンID（英数字）
 * @type string
 * @desc 例: yukino_h1 / rinne_end_happy
 *
 * @param commonEventId
 * @text コモンイベント
 * @type common_event
 *
 * @param nameJP
 * @text 表示名(JP)
 * @type string
 *
 * @param nameEN
 * @text 表示名(EN)
 * @type string
 *
 * @param nameZH
 * @text 表示名(ZH)
 * @type string
 *
 * @param nameKO
 * @text 表示名(KO)
 * @type string
 *
 * @param unlockViaSwitchId
 * @text 解放判定に使うスイッチ(任意)
 * @type switch
 * @default 0
 * @desc このスイッチがONなら未解放でも選択可（本編の既存フラグ流用したい場合に）
 */
(() => {
  const PN = "HS_RecallTitle";
  const P = PluginManager.parameters(PN);

  const param = {
    langVarId: Number(P.LanguageVariableId || 0),
    putInTitle: P.PutInTitle === "true",
    recallCmd: [
      String(P.RecallCommandJP || "回想"),
      String(P.RecallCommandEN || "Memories"),
      String(P.RecallCommandZH || "回想"),
      String(P.RecallCommandKO || "회상"),
    ],
    lockedText: [
      String(P.LockedTextJP || "？？？"),
      String(P.LockedTextEN || "????"),
      String(P.LockedTextZH || "？？？"),
      String(P.LockedTextKO || "？？？"),
    ],
    recallFlagSwitchId: Number(P.RecallFlagSwitchId || 0),
    recallMapId: Number(P.RecallMapId || 1),
    recallStartX: Number(P.RecallStartX || 0),
    recallStartY: Number(P.RecallStartY || 0),
    scenes: JSON.parse(P.Scenes || "[]").map(s => {
      const o = JSON.parse(s);
      return {
        id: String(o.id || ""),
        commonEventId: Number(o.commonEventId || 0),
        nameJP: String(o.nameJP || ""),
        nameEN: String(o.nameEN || ""),
        nameZH: String(o.nameZH || ""),
        nameKO: String(o.nameKO || ""),
        unlockViaSwitchId: Number(o.unlockViaSwitchId || 0),
      };
    })
  };

  // --------- 永続管理（ConfigManager） ---------
  const HSRecall = {
    unlocked: new Set(),
    loadFromConfig() {
      const a = ConfigManager.hsRecallIds || [];
      this.unlocked = new Set(a);
    },
    saveToConfig() {
      ConfigManager.hsRecallIds = Array.from(this.unlocked);
      ConfigManager.save();
    },
    unlockById(id) {
      if (!id) return;
      if (!this.unlocked.has(id)) {
        this.unlocked.add(id);
        this.saveToConfig();
      }
    },
    unlockByCommonEvent(ceId) {
      const sc = param.scenes.find(s => s.commonEventId === ceId);
      if (sc) this.unlockById(sc.id);
    },
    isUnlocked(scene) {
      if (scene.unlockViaSwitchId && $gameSwitches && $gameSwitches.value(scene.unlockViaSwitchId)) {
        return true;
      }
      return this.unlocked.has(scene.id);
    },
    currentLang() {
      try {
        // 0:JP,1:EN,2:ZH,3:KO
        if ($gameVariables && param.langVarId > 0) {
          const v = $gameVariables.value(param.langVarId) ?? 0;
          return Math.max(0, Math.min(3, Number(v) || 0));
        }
      } catch (e) {}
      return 0;
    },
    commandName() {
      return param.recallCmd[this.currentLang()] || param.recallCmd[0];
    },
    lockedString() {
      return param.lockedText[this.currentLang()] || param.lockedText[0];
    },
    localizedSceneName(scene) {
      const lang = this.currentLang();
      return [scene.nameJP, scene.nameEN, scene.nameZH, scene.nameKO][lang] || scene.nameJP || "";
    },
    play(scene) {
      // 新規ゲームを立ち上げて専用マップでコモンイベントを再生
      DataManager.setupNewGame();
      if (param.recallFlagSwitchId > 0) {
        $gameSwitches.setValue(param.recallFlagSwitchId, true);
      }
      if (param.recallMapId > 0) {
        $gamePlayer.reserveTransfer(param.recallMapId, param.recallStartX, param.recallStartY, 2, 0);
      }
      $gameTemp.reserveCommonEvent(scene.commonEventId);
      SceneManager.goto(Scene_Map);
    }
  };

  // ConfigManagerに項目を追加
  const _CM_make = ConfigManager.makeData;
  ConfigManager.makeData = function() {
    const data = _CM_make.call(this);
    data.hsRecallIds = this.hsRecallIds || [];
    return data;
  };
  const _CM_apply = ConfigManager.applyData;
  ConfigManager.applyData = function(config) {
    _CM_apply.call(this, config);
    this.hsRecallIds = config.hsRecallIds || [];
    HSRecall.loadFromConfig();
  };

  // --------- プラグインコマンド ---------
  PluginManager.registerCommand(PN, "UnlockScene", args => {
    HSRecall.unlockById(String(args.id || ""));
  });
  PluginManager.registerCommand(PN, "UnlockByCommonEvent", args => {
    HSRecall.unlockByCommonEvent(Number(args.commonEventId || 0));
  });
  PluginManager.registerCommand(PN, "OpenRecall", () => {
    SceneManager.push(Scene_HSRecall);
  });
  PluginManager.registerCommand(PN, "ExitToTitleFromRecall", () => {
    // 回想フラグを下げてタイトルへ
    if (param.recallFlagSwitchId > 0) $gameSwitches.setValue(param.recallFlagSwitchId, false);
    SceneManager.goto(Scene_Title);
  });

  // --------- タイトルにコマンド追加 ---------
  if (param.putInTitle) {
    const _WinTitle_make = Window_TitleCommand.prototype.makeCommandList;
    Window_TitleCommand.prototype.makeCommandList = function() {
      _WinTitle_make.call(this);
      this.addCommand(HSRecall.commandName(), "hs_recall");
    };
    const _SceneTitle_createCmd = Scene_Title.prototype.createCommandWindow;
    Scene_Title.prototype.createCommandWindow = function() {
      _SceneTitle_createCmd.call(this);
      this._commandWindow.setHandler("hs_recall", this.commandHsRecall.bind(this));
    };
    Scene_Title.prototype.commandHsRecall = function() {
      this._commandWindow.close();
      SceneManager.push(Scene_HSRecall);
    };
  }

  // --------- 回想シーン ---------
  function Scene_HSRecall() { this.initialize(...arguments); }
  Scene_HSRecall.prototype = Object.create(Scene_Base.prototype);
  Scene_HSRecall.prototype.constructor = Scene_HSRecall;

  Scene_HSRecall.prototype.initialize = function() {
    Scene_Base.prototype.initialize.call(this);
  };
  Scene_HSRecall.prototype.create = function() {
    Scene_Base.prototype.create.call(this);
    this.createWindowLayer();
    const ww = Math.floor(Graphics.boxWidth * 0.8);
    const wh = Math.floor(Graphics.boxHeight * 0.8);
    const wx = Math.floor((Graphics.boxWidth - ww) / 2);
    const wy = Math.floor((Graphics.boxHeight - wh) / 2);

    this._listWindow = new Window_HSRecallList(new Rectangle(wx, wy, ww, wh));
    this._listWindow.setHandler("ok", this.onOk.bind(this));
    this._listWindow.setHandler("cancel", this.popScene.bind(this));
    this.addWindow(this._listWindow);
  };
  Scene_HSRecall.prototype.onOk = function() {
    const scene = this._listWindow.currentSceneData();
    if (!scene) return;
    HSRecall.play(scene);
  };

  function Window_HSRecallList() { this.initialize(...arguments); }
  Window_HSRecallList.prototype = Object.create(Window_Selectable.prototype);
  Window_HSRecallList.prototype.constructor = Window_HSRecallList;

  Window_HSRecallList.prototype.initialize = function(rect) {
    Window_Selectable.prototype.initialize.call(this, rect);
    this.refresh();
    this.activate();
    this.select(0);
  };
  Window_HSRecallList.prototype.maxItems = function() { return param.scenes.length; };
  Window_HSRecallList.prototype.itemHeight = function() { return Math.floor(this.lineHeight() + 8); };
  Window_HSRecallList.prototype.isCurrentItemEnabled = function() {
    const s = param.scenes[this.index()];
    return !!s && HSRecall.isUnlocked(s);
  };
  Window_HSRecallList.prototype.currentSceneData = function() {
    const s = param.scenes[this.index()];
    return s && HSRecall.isUnlocked(s) ? s : null;
  };
  Window_HSRecallList.prototype.drawItem = function(index) {
    const rect = this.itemLineRect(index);
    const s = param.scenes[index];
    if (!s) return;
    const unlocked = HSRecall.isUnlocked(s);
    const name = unlocked ? HSRecall.localizedSceneName(s) : HSRecall.lockedString();
    this.changePaintOpacity(unlocked);
    this.drawText(name, rect.x, rect.y, rect.width);
    this.changePaintOpacity(true);
  };
})();
